<?php
/**
 * Skripta za automatsku detekciju dimenzija
 * 
 * Datoteka: detect_dimensions.php
 * Opis: Poziva Python skriptu za detekciju dimenzija i vraća rezultate
 */

// Postavljanje zaglavlja za JSON odgovor
header('Content-Type: application/json');

// Provjera zahtjeva
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode([
        'success' => false,
        'message' => 'Nevažeći zahtjev.'
    ]);
    exit;
}

// Dohvaćanje podataka iz POST zahtjeva
$data = json_decode(file_get_contents('php://input'), true);

if (!isset($data['file']) || empty($data['file'])) {
    echo json_encode([
        'success' => false,
        'message' => 'Nije specificirana datoteka.'
    ]);
    exit;
}

// Definiranje putanja - koristimo apsolutne putanje za veću sigurnost
$rootPath = '/var/www/html/';
$uploadDir = $rootPath . 'uploads/';
$logsPath = $rootPath . 'logs/';
$filePath = $uploadDir . $data['file'];

// Stvaranje potrebnih direktorija ako ne postoje
if (!file_exists($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

if (!file_exists($logsPath)) {
    mkdir($logsPath, 0755, true);
}

// Provjera postoji li datoteka
if (!file_exists($filePath)) {
    echo json_encode([
        'success' => false,
        'message' => 'Datoteka ne postoji.'
    ]);
    exit;
}

// Dobivanje informacija o datoteci
$fileInfo = pathinfo($filePath);
$extension = strtolower($fileInfo['extension']);

// Priprema putanja za označeni PDF i Excel
$timestamp = time();
$markedPdfPath = $uploadDir . $timestamp . '_marked_' . basename($filePath);
$excelPath = $uploadDir . $timestamp . '_dimensions.xlsx';

// Ako je datoteka PDF, poziva Python skriptu za analizu
if ($extension === 'pdf') {
    // Putanja do Python skripte i interpretera
    $pythonScript = $rootPath . 'improved_analyzer.py';
    $pythonInterpreter = '/usr/bin/env python3';
    
    // Komanda za poziv Python skripte
    $command = $pythonInterpreter . ' ' . escapeshellarg($pythonScript) .
               ' --file_path ' . escapeshellarg($filePath) .
               ' --output_pdf ' . escapeshellarg($markedPdfPath) .
               ' --output_excel ' . escapeshellarg($excelPath);
    
    // Sigurno izvršavanje komande
    $command = escapeshellcmd($command);
    
    // Logiranje komande
    file_put_contents(
        $logsPath . 'detect_dimensions.log',
        date('Y-m-d H:i:s') . " Executing: $command\n",
        FILE_APPEND
    );
    
    // Izvršavanje komande i hvatanje outputa
    $output = null;
    $returnCode = null;
    exec($command . ' 2>&1', $output, $returnCode);
    $outputStr = implode("\n", $output);
    
    // Logiranje izlaza i return koda
    file_put_contents(
        $logsPath . 'detect_dimensions.log',
        date('Y-m-d H:i:s') . " Return code: $returnCode\nOutput: $outputStr\n\n",
        FILE_APPEND
    );
    
    // Provjera rezultata
    $result = @json_decode($outputStr, true);
    
    if ($result && isset($result['success']) && $result['success'] === true && isset($result['data'])) {
        // Kategorizacija dimenzija
        $categories = categorizeDimensions($result['data']);
        
        // Zabilježi u bazu podataka ako je potrebno
        // saveToDatabase($filePath, $markedPdfPath, $excelPath, $result['data']);
        
        echo json_encode([
            'success' => true,
            'message' => 'Dimenzije uspješno detektirane.',
            'file' => [
                'original' => basename($filePath),
                'marked' => basename($markedPdfPath),
                'excel' => basename($excelPath)
            ],
            'dimensions' => $result['data'],
            'categories' => $categories
        ]);
    } else {
        // Provjera je li datoteka stvorena unatoč grešci
        $outputFilesExist = file_exists($markedPdfPath) && file_exists($excelPath);
        
        echo json_encode([
            'success' => false,
            'message' => 'Greška prilikom detekcije dimenzija.',
            'error' => isset($result['error']) ? $result['error'] : 'Nepoznata greška',
            'command_output' => $outputStr,
            'return_code' => $returnCode,
            'output_files_exist' => $outputFilesExist
        ]);
    }
} else {
    // Za ne-PDF datoteke, koristi alternativnu metodu detekcije
    $sampleDimensions = simulateDimensions();
    $categories = categorizeDimensions($sampleDimensions);
    
    echo json_encode([
        'success' => true,
        'message' => 'Dimenzije uspješno detektirane na slici.',
        'file' => [
            'original' => basename($filePath)
        ],
        'dimensions' => $sampleDimensions,
        'categories' => $categories
    ]);
}

/**
 * Kategorizira dimenzije prema njihovom tipu
 * 
 * @param array $dimensions Lista dimenzija
 * @return array Kategorizirane dimenzije
 */
function categorizeDimensions($dimensions) {
    $categories = [
        'dimension' => [],
        'thread' => [],
        'roughness' => [],
        'gdt' => []
    ];
    
    foreach ($dimensions as $dim) {
        // Prepoznavanje tipa dimenzije prema tekstu
        $text = isset($dim['text']) ? $dim['text'] : '';
        $type = 'dimension'; // Zadani tip
        $subtype = '';
        
        // Prepoznavanje navoja
        if (preg_match('/M\d+(\.\d+)?\s*[×x]\s*\d+(\.\d+)?/', $text)) {
            $type = 'thread';
            $subtype = 'metric';
        }
        // Prepoznavanje promjera
        elseif (preg_match('/Ø|ø/', $text)) {
            $type = 'dimension';
            $subtype = 'diameter';
        }
        // Prepoznavanje hrapavosti
        elseif (preg_match('/Ra|▽/', $text)) {
            $type = 'roughness';
            $subtype = 'ra';
        }
        // Prepoznavanje GD&T
        elseif (preg_match('/[\[\|⟂◎⌭\]]/', $text)) {
            $type = 'gdt';
            // Podtip GD&T (okomitost, paralelnost itd.)
            if (preg_match('/⟂/', $text)) {
                $subtype = 'perpendicularity';
            } elseif (preg_match('/↗/', $text)) {
                $subtype = 'angularity';
            } elseif (preg_match('/◎/', $text)) {
                $subtype = 'concentricity';
            } else {
                $subtype = 'other';
            }
        }
        
        // Dodaj kategorizirane dimenzije
        $dim['type'] = $type;
        $dim['subtype'] = $subtype;
        $categories[$type][] = $dim;
    }
    
    return $categories;
}

/**
 * Simulira listu dimenzija za ne-PDF datoteke (za demonstraciju)
 * 
 * @return array Simulirane dimenzije
 */
function simulateDimensions() {
    $dimensions = [];
    $sampleTexts = [
        "Ø28 k6 (+0.015/+0.002)",
        "Ø13",
        "Ø44.2 h7 (0/-0.025)",
        "15.46",
        "20.55",
        "M10×1.5---6g/6H",
        "▽√URa 3.2",
        "[⟂|0.05|A-B]"
    ];
    
    for ($i = 0; $i < count($sampleTexts); $i++) {
        $dimensions[] = [
            'number' => $i + 1,
            'text' => $sampleTexts[$i],
            'page' => 1,
            'x0' => rand(50, 500),
            'y0' => rand(50, 500),
            'x1' => rand(100, 550),
            'y1' => rand(100, 550)
        ];
    }
    
    return $dimensions;
}

/**
 * Sprema rezultate u bazu podataka
 * Ovo je priprema za buduće integracije s bazom podataka
 */
/*
function saveToDatabase($originalFile, $markedFile, $excelFile, $dimensions) {
    // Primjer kako bi se dimenzije spremile u bazu podataka
    // Dodajte ovu funkcionalnost kasnije kada implementirate bazu podataka
    global $rootPath;
    
    // Uključi konfiguraciju baze
    require_once $rootPath . 'db_config.php';
    
    // Dodaj zapis o dokumentu
    $stmt = $conn->prepare("INSERT INTO documents (original_name, file_path, file_type, upload_date, marked_file_path, excel_file_path) VALUES (?, ?, ?, NOW(), ?, ?)");
    $originalName = basename($originalFile);
    $filePath = str_replace($rootPath, '', $originalFile);
    $fileType = pathinfo($originalFile, PATHINFO_EXTENSION);
    $markedPath = str_replace($rootPath, '', $markedFile);
    $excelPath = str_replace($rootPath, '', $excelFile);
    
    $stmt->bind_param("sssss", $originalName, $filePath, $fileType, $markedPath, $excelPath);
    $stmt->execute();
    
    $documentId = $stmt->insert_id;
    
    // Dodaj dimenzije
    $stmtDim = $conn->prepare("INSERT INTO dimensions (document_id, number, text, type, subtype, page, x0, y0, x1, y1) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
    
    foreach ($dimensions as $dim) {
        $number = $dim['number'];
        $text = $dim['text'];
        $type = $dim['type'];
        $subtype = $dim['subtype'] ?? null;
        $page = $dim['page'] ?? 1;
        $x0 = $dim['x0'] ?? 0;
        $y0 = $dim['y0'] ?? 0;
        $x1 = $dim['x1'] ?? 0;
        $y1 = $dim['y1'] ?? 0;
        
        $stmtDim->bind_param("iisssiiddd", $documentId, $number, $text, $type, $subtype, $page, $x0, $y0, $x1, $y1);
        $stmtDim->execute();
    }
    
    $stmt->close();
    $stmtDim->close();
    $conn->close();
}
*/