<?php
/**
 * Skripta za preuzimanje datoteka
 * 
 * Datoteka: download.php
 * Opis: Omogućuje sigurno preuzimanje generiranih dokumenata (PDF, Excel)
 * iz uploads direktorija
 * 
 * @version 1.0
 */

// Definiranje osnovnih putanja za Linux hosting
$rootPath = '/var/www/html/';
$uploadsPath = $rootPath . 'uploads/';
$logsPath = $rootPath . 'logs/';

// Uključi konfiguraciju baze podataka ako je potrebna provjera iz baze
// require_once $rootPath . 'db_config.php';

// Započni sesiju ako je potrebna provjera sesije/korisnika
// session_start();

// Provjera je li datoteka specificirana
if (!isset($_GET['file']) || empty($_GET['file'])) {
    header('HTTP/1.0 400 Bad Request');
    echo 'Datoteka nije specificirana.';
    exit;
}

// Sanitizacija imena datoteke (uklanjanje posebnih znakova i putanja)
$requestedFile = $_GET['file'];
$fileName = basename($requestedFile);

// Dodatna sigurnosna provjera - ne dopuštamo pristup drugim direktorijima
if (strpos($fileName, '..') !== false || strpos($fileName, '/') !== false || strpos($fileName, '\\') !== false) {
    header('HTTP/1.0 403 Forbidden');
    echo 'Pristup odbijen.';
    exit;
}

// Provjera dozvoljenih ekstenzija
$extension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
$allowedExtensions = ['pdf', 'xlsx', 'jpg', 'jpeg', 'png', 'tiff', 'bmp'];

if (!in_array($extension, $allowedExtensions)) {
    header('HTTP/1.0 403 Forbidden');
    echo 'Nedozvoljena vrsta datoteke.';
    exit;
}

// Putanja do datoteke u uploads direktoriju - koristimo apsolutnu putanju
$filePath = $uploadsPath . $fileName;

// Provjera postoji li datoteka
if (!file_exists($filePath)) {
    header('HTTP/1.0 404 Not Found');
    echo 'Datoteka ne postoji.';
    exit;
}

// Provjera je li datoteka unutar uploads direktorija (sigurnosna mjera)
$realPath = realpath($filePath);
$uploadsDir = realpath($uploadsPath);

if (strpos($realPath, $uploadsDir) !== 0) {
    header('HTTP/1.0 403 Forbidden');
    echo 'Pristup odbijen.';
    exit;
}

// Ovdje možete dodati provjeru prava pristupa datoteci iz baze podataka
// Npr. provjera pripada li datoteka trenutnom korisniku
/*
if (isset($_SESSION['user_id'])) {
    $userId = $_SESSION['user_id'];
    $fileId = explode('_', $fileName)[0]; // Pretpostavka da je ID u imenu datoteke
    
    // Provjera u bazi
    $stmt = $conn->prepare("SELECT id FROM documents WHERE id = ? AND user_id = ?");
    $stmt->bind_param("ii", $fileId, $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        header('HTTP/1.0 403 Forbidden');
        echo 'Nemate pravo pristupa ovoj datoteci.';
        exit;
    }
}
*/

// Određivanje MIME tipa
$mimeTypes = [
    'pdf'  => 'application/pdf',
    'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
    'jpg'  => 'image/jpeg',
    'jpeg' => 'image/jpeg',
    'png'  => 'image/png',
    'tiff' => 'image/tiff',
    'bmp'  => 'image/bmp'
];

$contentType = isset($mimeTypes[$extension]) ? $mimeTypes[$extension] : 'application/octet-stream';

// Provjera veličine datoteke
$fileSize = filesize($filePath);
if ($fileSize === false) {
    header('HTTP/1.0 500 Internal Server Error');
    echo 'Greška prilikom određivanja veličine datoteke.';
    exit;
}

// Postavljanje zaglavlja
header('Content-Type: ' . $contentType);
header('Content-Disposition: attachment; filename="' . $fileName . '"');
header('Content-Length: ' . $fileSize);
header('Cache-Control: private, max-age=0, must-revalidate');
header('Pragma: public');
header('Expires: 0');

// Dodatna zaštita - onemogući prikaz PHP grešaka
ini_set('display_errors', 0);

// Stvaranje logs direktorija ako ne postoji
if (!is_dir($logsPath)) {
    mkdir($logsPath, 0755, true);
}

// Zabilježi preuzimanje u log datoteku
$logMessage = date('Y-m-d H:i:s') . ' - Downloaded: ' . $fileName;
if (isset($_SERVER['REMOTE_ADDR'])) {
    $logMessage .= ' from IP: ' . $_SERVER['REMOTE_ADDR'];
}
error_log($logMessage . PHP_EOL, 3, $logsPath . 'downloads.log');

// Pročitaj i pošalji datoteku korisniku
if ($fileSize > 1024 * 1024) {
    // Za veće datoteke - šalji u dijelovima
    $handle = fopen($filePath, 'rb');
    if ($handle === false) {
        header('HTTP/1.0 500 Internal Server Error');
        echo 'Greška prilikom čitanja datoteke.';
        exit;
    }
    
    while (!feof($handle)) {
        echo fread($handle, 1024 * 8); // Šalji po 8KB
        flush();
    }
    fclose($handle);
} else {
    // Za manje datoteke - šalji odjednom
    readfile($filePath);
}

exit;