<?php
/**
 * Skripta za izvoz dimenzija u Excel
 * 
 * Datoteka: export.php
 * Opis: Izvozi dimenzije u Excel dokument prema kategorijama
 */

// Nužno je imati PhpSpreadsheet biblioteku instaliranu preko Composera
// composer require phpoffice/phpspreadsheet

require 'vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

// Postavljanje zaglavlja za JSON odgovor
header('Content-Type: application/json');

// Provjera zahtjeva
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode([
        'success' => false,
        'message' => 'Nevažeći zahtjev.'
    ]);
    exit;
}

// Dohvaćanje podataka iz POST zahtjeva
$data = json_decode(file_get_contents('php://input'), true);

if (!isset($data['dimensions']) || !is_array($data['dimensions'])) {
    echo json_encode([
        'success' => false,
        'message' => 'Nevažeći podaci.'
    ]);
    exit;
}

try {
    // Stvaranje nove Excel datoteke
    $spreadsheet = new Spreadsheet();
    
    // Postavljanje metapodataka
    $spreadsheet->getProperties()
        ->setCreator('Detektor dimenzija')
        ->setLastModifiedBy('Detektor dimenzija')
        ->setTitle('Izvještaj dimenzija')
        ->setSubject('Tehnički nacrt - dimenzije')
        ->setDescription('Izvještaj detektiranih dimenzija na tehničkom nacrtu')
        ->setKeywords('dimenzije nacrt tehnički')
        ->setCategory('Izvještaj');
    
    // Kategoriziranje dimenzija
    $categories = [
        'dimension' => [
            'name' => 'Dimenzije',
            'items' => []
        ],
        'thread' => [
            'name' => 'Navoji',
            'items' => []
        ],
        'roughness' => [
            'name' => 'Hrapavost',
            'items' => []
        ],
        'gdt' => [
            'name' => 'GD&T',
            'items' => []
        ]
    ];
    
    // Razvrstavanje dimenzija u kategorije
    foreach ($data['dimensions'] as $dim) {
        if (isset($dim['type']) && isset($categories[$dim['type']])) {
            $categories[$dim['type']]['items'][] = $dim;
        } else {
            // Ako tip nije definiran, stavi u općenite dimenzije
            $categories['dimension']['items'][] = $dim;
        }
    }
    
    // Kreiranje sheet-a za svaku kategoriju
    $categoryIndex = 0;
    foreach ($categories as $category => $categoryData) {
        if (count($categoryData['items']) > 0) {
            // Ako je prvi sheet, uzimamo aktivni, inače stvaramo novi
            if ($categoryIndex > 0) {
                $spreadsheet->createSheet();
            }
            
            $sheet = $spreadsheet->setActiveSheetIndex($categoryIndex);
            $sheet->setTitle($categoryData['name']);
            
            // Postavljanje zaglavlja
            $sheet->setCellValue('A1', 'Broj');
            $sheet->setCellValue('B1', 'Tekst');
            $sheet->setCellValue('C1', 'Podkategorija');
            $sheet->setCellValue('D1', 'Stranica');
            
            // Stiliziranje zaglavlja
            $sheet->getStyle('A1:D1')->getFont()->setBold(true);
            $sheet->getStyle('A1:D1')->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('4472C4');
            $sheet->getStyle('A1:D1')->getFont()->getColor()->setRGB('FFFFFF');
            $sheet->getStyle('A1:D1')->getBorders()->getAllBorders()->setBorderStyle(Border::BORDER_THIN);
            $sheet->getStyle('A1:D1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
            
            // Popunjavanje podataka
            $row = 2;
            foreach ($categoryData['items'] as $item) {
                $sheet->setCellValue('A' . $row, $item['number']);
                $sheet->setCellValue('B' . $row, $item['text']);
                $sheet->setCellValue('C' . $row, isset($item['subtype']) ? ucfirst($item['subtype']) : '');
                $sheet->setCellValue('D' . $row, isset($item['page']) ? $item['page'] : 1);
                
                // Stiliziranje retka
                $sheet->getStyle('A' . $row . ':D' . $row)->getBorders()->getAllBorders()->setBorderStyle(Border::BORDER_THIN);
                if ($row % 2 == 0) {
                    $sheet->getStyle('A' . $row . ':D' . $row)->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('E9EDF7');
                }
                
                $row++;
            }
            
            // Podešavanje širina stupaca
            $sheet->getColumnDimension('A')->setWidth(10);
            $sheet->getColumnDimension('B')->setWidth(30);
            $sheet->getColumnDimension('C')->setWidth(20);
            $sheet->getColumnDimension('D')->setWidth(10);
            
            $categoryIndex++;
        }
    }
    
    // Stvaranje sheet-a sa svim dimenzijama
    if (count($data['dimensions']) > 0) {
        $spreadsheet->createSheet();
        $sheet = $spreadsheet->setActiveSheetIndex($categoryIndex);
        $sheet->setTitle('Sve dimenzije');
        
        // Postavljanje zaglavlja
        $sheet->setCellValue('A1', 'Broj');
        $sheet->setCellValue('B1', 'Tekst');
        $sheet->setCellValue('C1', 'Kategorija');
        $sheet->setCellValue('D1', 'Podkategorija');
        $sheet->setCellValue('E1', 'Stranica');
        
        // Stiliziranje zaglavlja
        $sheet->getStyle('A1:E1')->getFont()->setBold(true);
        $sheet->getStyle('A1:E1')->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('4472C4');
        $sheet->getStyle('A1:E1')->getFont()->getColor()->setRGB('FFFFFF');
        $sheet->getStyle('A1:E1')->getBorders()->getAllBorders()->setBorderStyle(Border::BORDER_THIN);
        $sheet->getStyle('A1:E1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
        
        // Popunjavanje podataka
        $row = 2;
        foreach ($data['dimensions'] as $item) {
            $sheet->setCellValue('A' . $row, $item['number']);
            $sheet->setCellValue('B' . $row, $item['text']);
            $sheet->setCellValue('C' . $row, isset($item['type']) ? ucfirst($item['type']) : 'Dimension');
            $sheet->setCellValue('D' . $row, isset($item['subtype']) ? ucfirst($item['subtype']) : '');
            $sheet->setCellValue('E' . $row, isset($item['page']) ? $item['page'] : 1);
            
            // Stiliziranje retka
            $sheet->getStyle('A' . $row . ':E' . $row)->getBorders()->getAllBorders()->setBorderStyle(Border::BORDER_THIN);
            if ($row % 2 == 0) {
                $sheet->getStyle('A' . $row . ':E' . $row)->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('E9EDF7');
            }
            
            $row++;
        }
        
        // Podešavanje širina stupaca
        $sheet->getColumnDimension('A')->setWidth(10);
        $sheet->getColumnDimension('B')->setWidth(30);
        $sheet->getColumnDimension('C')->setWidth(15);
        $sheet->getColumnDimension('D')->setWidth(20);
        $sheet->getColumnDimension('E')->setWidth(10);
    }
    
    // Postavljanje prvog sheet-a kao aktivnog
    $spreadsheet->setActiveSheetIndex(0);
    
    // Definiranje apsolutne putanje za Linux hosting
    $rootPath = '/var/www/html/';
    $uploadDir = $rootPath . 'uploads/';
    $fileName = time() . '_dimensions_export.xlsx';
    $filePath = $uploadDir . $fileName;
    
    // Stvaranje direktorija ako ne postoji
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    // Spremanje Excel datoteke
    $writer = new Xlsx($spreadsheet);
    $writer->save($filePath);
    
    // Odgovor s putanjom do Excel datoteke
    echo json_encode([
        'success' => true,
        'message' => 'Excel datoteka uspješno generirana.',
        'file' => basename($filePath),
        'download_url' => 'download.php?file=' . basename($filePath)
    ]);
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Greška prilikom generiranja Excel datoteke.',
        'error' => $e->getMessage()
    ]);
}