<?php
/**
 * Skripta za ručno dodavanje dimenzija
 * 
 * Datoteka: manual_add.php
 * Opis: Obrađuje ručno dodane dimenzije i ažurira PDF s oznakama
 */

// Postavljanje zaglavlja za JSON odgovor
header('Content-Type: application/json');

// Provjera zahtjeva
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode([
        'success' => false,
        'message' => 'Nevažeći zahtjev.'
    ]);
    exit;
}

// Dohvaćanje podataka iz POST zahtjeva
$data = json_decode(file_get_contents('php://input'), true);

if (!isset($data['dimensions']) || !is_array($data['dimensions']) || !isset($data['file'])) {
    echo json_encode([
        'success' => false,
        'message' => 'Nevažeći podaci.'
    ]);
    exit;
}

// Definiranje putanja - koristimo apsolutne putanje za veću sigurnost
$rootPath = '/var/www/html/';
$uploadDir = $rootPath . 'uploads/';
$filePath = $uploadDir . $data['file'];

// Provjera postoji li uploads direktorij, ako ne - stvori ga
if (!file_exists($uploadDir)) {
    if (!mkdir($uploadDir, 0755, true)) {
        echo json_encode([
            'success' => false,
            'message' => 'Nije moguće stvoriti direktorij za upload.'
        ]);
        exit;
    }
}

// Provjera postoji li datoteka
if (!file_exists($filePath)) {
    echo json_encode([
        'success' => false,
        'message' => 'Datoteka ne postoji.'
    ]);
    exit;
}

// Dobivanje informacija o datoteci
$fileInfo = pathinfo($filePath);
$extension = strtolower($fileInfo['extension']);

// Priprema putanja za označeni PDF i Excel
$timestamp = time();
$markedPdfPath = $uploadDir . $timestamp . '_manual_marked_' . basename($filePath);
$excelPath = $uploadDir . $timestamp . '_manual_dimensions.xlsx';

// Priprema JSON dimenzija za Python skriptu
$dimensionsJson = json_encode($data['dimensions']);

// Ako je datoteka PDF, poziva Python skriptu za označavanje
if ($extension === 'pdf') {
    // Putanja do Python skripte
    $pythonScript = $rootPath . 'improved_analyzer.py';
    
    // Komanda za poziv Python skripte s ručnim načinom
    // Ispravno korištenje Python interpretera na Linux sustavu
    $pythonInterpreter = '/usr/bin/env python3';
    $command = escapeshellcmd($pythonInterpreter . ' ' . 
               escapeshellarg($pythonScript) .
               ' --file_path ' . escapeshellarg($filePath) .
               ' --output_pdf ' . escapeshellarg($markedPdfPath) .
               ' --output_excel ' . escapeshellarg($excelPath) .
               ' --manual_mode' .
               ' --dimensions ' . escapeshellarg($dimensionsJson));
    
    // Izvršavanje komande
    $output = shell_exec($command);
    
    // Zapisivanje u log u slučaju potrebe za debugiranjem
    $logPath = $rootPath . 'logs/';
    if (!file_exists($logPath)) {
        mkdir($logPath, 0755, true);
    }
    
    file_put_contents(
        $logPath . 'manual_add.log', 
        date('Y-m-d H:i:s') . " Command: $command\nOutput: $output\n\n", 
        FILE_APPEND
    );
    
    // Provjera rezultata
    $result = json_decode($output, true);
    
    if ($result && $result['success']) {
        echo json_encode([
            'success' => true,
            'message' => 'Dimenzije uspješno dodane.',
            'file' => [
                'original' => basename($filePath),
                'marked' => basename($markedPdfPath),
                'excel' => basename($excelPath)
            ],
            'dimensions' => $data['dimensions']
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'Greška prilikom označavanja dimenzija.',
            'error' => isset($result['error']) ? $result['error'] : 'Nepoznata greška',
            'command_output' => $output
        ]);
    }
} else {
    // Za slike, jednostavno vrati uspjeh (u pravoj implementaciji bi trebalo implementirati označavanje na slikama)
    echo json_encode([
        'success' => true,
        'message' => 'Dimenzije uspješno dodane za sliku.',
        'file' => [
            'original' => basename($filePath)
        ],
        'dimensions' => $data['dimensions']
    ]);
}