<?php
/**
 * Skripta za upload i obradu dokumenata
 * 
 * Datoteka: upload.php
 * Opis: Obrađuje upload dokumenata i poziva Python skriptu za analizu
 */

// Postavljanje zaglavlja za JSON odgovor
header('Content-Type: application/json');

// Definiranje putanja - koristimo apsolutne putanje za veću sigurnost
$rootPath = '/var/www/html/';
$uploadDir = $rootPath . 'uploads/';
$logsPath = $rootPath . 'logs/';
$allowedExtensions = ['pdf', 'jpg', 'jpeg', 'png', 'tiff', 'bmp'];
$maxFileSize = 10 * 1024 * 1024; // 10 MB

// Stvaranje direktorija za upload ako ne postoji
if (!file_exists($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

// Stvaranje direktorija za logove ako ne postoji
if (!file_exists($logsPath)) {
    mkdir($logsPath, 0755, true);
}

// Provjera zahtjeva
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode([
        'success' => false,
        'message' => 'Nevažeći zahtjev.'
    ]);
    exit;
}

// Provjera je li datoteka uploadana
if (!isset($_FILES['documentFile']) || $_FILES['documentFile']['error'] !== UPLOAD_ERR_OK) {
    $error = '';
    switch ($_FILES['documentFile']['error']) {
        case UPLOAD_ERR_INI_SIZE:
        case UPLOAD_ERR_FORM_SIZE:
            $error = 'Datoteka je prevelika.';
            break;
        case UPLOAD_ERR_PARTIAL:
            $error = 'Datoteka je djelomično uploadana.';
            break;
        case UPLOAD_ERR_NO_FILE:
            $error = 'Nijedna datoteka nije uploadana.';
            break;
        default:
            $error = 'Greška prilikom uploada datoteke.';
    }
    
    echo json_encode([
        'success' => false,
        'message' => $error
    ]);
    exit;
}

// Provjera veličine datoteke
if ($_FILES['documentFile']['size'] > $maxFileSize) {
    echo json_encode([
        'success' => false,
        'message' => 'Datoteka ne smije biti veća od 10 MB.'
    ]);
    exit;
}

// Provjera ekstenzije datoteke
$fileInfo = pathinfo($_FILES['documentFile']['name']);
$extension = strtolower($fileInfo['extension']);

if (!in_array($extension, $allowedExtensions)) {
    echo json_encode([
        'success' => false,
        'message' => 'Nevažeći format datoteke. Dozvoljeni formati: PDF, JPEG, PNG, TIFF, BMP.'
    ]);
    exit;
}

// Generiranje jedinstvenog imena datoteke
$timestamp = time();
$fileName = $timestamp . '_' . basename($_FILES['documentFile']['name']);
$filePath = $uploadDir . $fileName;

// Upload datoteke
if (!move_uploaded_file($_FILES['documentFile']['tmp_name'], $filePath)) {
    echo json_encode([
        'success' => false,
        'message' => 'Greška prilikom spremanja datoteke.'
    ]);
    exit;
}

// Priprema putanje za označeni PDF i Excel
$markedPdfPath = $uploadDir . $timestamp . '_marked_' . basename($_FILES['documentFile']['name']);
$excelPath = $uploadDir . $timestamp . '_dimensions.xlsx';

// Ako je datoteka PDF, poziva Python skriptu za analizu
if ($extension === 'pdf') {
    // Putanja do Python skripte i interpretera
    $pythonScript = $rootPath . 'improved_analyzer.py';
    $pythonInterpreter = '/usr/bin/env python3';
    
    // Komanda za poziv Python skripte
    $command = escapeshellcmd($pythonInterpreter . ' ' . 
               escapeshellarg($pythonScript) .
               ' --file_path ' . escapeshellarg($filePath) .
               ' --output_pdf ' . escapeshellarg($markedPdfPath) .
               ' --output_excel ' . escapeshellarg($excelPath));
    
    // Zapisivanje komande u log
    file_put_contents(
        $logsPath . 'upload.log', 
        date('Y-m-d H:i:s') . " Executing: $command\n", 
        FILE_APPEND
    );
    
    // Izvršavanje komande
    $output = shell_exec($command);
    
    // Zapisivanje izlaza u log
    file_put_contents(
        $logsPath . 'upload.log', 
        date('Y-m-d H:i:s') . " Output: $output\n\n", 
        FILE_APPEND
    );
    
    // Provjera rezultata
    $result = json_decode($output, true);
    
    if ($result && $result['success']) {
        echo json_encode([
            'success' => true,
            'message' => 'Datoteka uspješno uploadana i analizirana.',
            'file' => [
                'original' => basename($filePath),
                'marked' => basename($markedPdfPath),
                'excel' => basename($excelPath)
            ],
            'dimensions' => $result['data']
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'Greška prilikom analize datoteke.',
            'file' => basename($filePath),
            'error' => isset($result['error']) ? $result['error'] : 'Nepoznata greška',
            'command_output' => $output
        ]);
    }
} else {
    // Za ne-PDF datoteke, samo vrati uspjeh
    echo json_encode([
        'success' => true,
        'message' => 'Datoteka uspješno uploadana.',
        'file' => basename($filePath)
    ]);
}